"""Functions for computing fundamental and adjoint string tensions.

This module defines a simplified model for estimating string tensions in
SU(N) gauge theories given a one‑dimensional kernel.  The real
simulation in the original ``vol4‑wilson‑loop‑adjoint`` repository
measured Wilson loops on a lattice; in the absence of that code we
approximate the behaviour as follows:

* **Fundamental string tension.**  The kernel values are raised to
  an exponent ``k`` and then averaged.  Multiplying by the coupling
  ``b`` yields a tension ``σ_fund``.  For ``k = 1`` this recovers the
  mean of the kernel up to the overall coupling.

* **Adjoint string tension.**  For an SU(N) gauge theory the Casimir
  scaling hypothesis suggests that the adjoint string tension is
  proportional to the fundamental tension by the ratio of Casimir
  invariants,

      σ_adj / σ_fund = (2 N²) / (N² − 1).

  For SU(3) this factor is 2.25.  The ``adjoint_string_tension``
  function implements this scaling.
"""

from typing import Sequence
import numpy as np


def fundamental_string_tension(kernel: np.ndarray, b: float, k_exp: float) -> float:
    """Compute the fundamental string tension.

    Parameters
    ----------
    kernel : numpy.ndarray
        One‑dimensional kernel array produced by ``make_kernel_from_eigs.py``.
    b : float
        Coupling constant controlling the overall strength of the tension.
    k_exp : float
        Exponent applied element‑wise to the kernel prior to averaging.

    Returns
    -------
    float
        The estimated fundamental string tension ``σ_fund``.
    """
    # Raise kernel values to the exponent and compute their mean.
    powered = kernel.astype(float) ** k_exp
    avg = np.mean(powered)
    return float(b * avg)


def adjoint_string_tension(sigma_fund: float, *, N: int = 3) -> float:
    """Compute the adjoint string tension from the fundamental tension.

    Parameters
    ----------
    sigma_fund : float
        The fundamental string tension ``σ_fund``.
    N : int, optional
        The number of colours in the SU(N) gauge group.  Defaults to 3.

    Returns
    -------
    float
        The estimated adjoint string tension ``σ_adj``.
    """
    # Casimir ratio: 2 N^2 / (N^2 - 1)
    numerator = 2 * (N ** 2)
    denominator = (N ** 2) - 1
    ratio = numerator / denominator
    return float(ratio * sigma_fund)
